<?php

class Shopware_Plugins_Frontend_CfCookieFirst_Bootstrap extends Shopware_Components_Plugin_Bootstrap
{
    private $pluginInfo = array();

    /**
     * Helper for cache array
     * @return array
     */
    private function getInvalidateCacheArray()
    {
        return array(
            'config', 'template', 'theme'
        );
    }

    /**
     * Helper for plugin info array
     * @return array
     */
    public function getPluginDetails(){
        if( empty($this->pluginInfo) )
            $this->pluginInfo = json_decode( file_get_contents(__DIR__ . '/plugin.json'), true );

        return $this->pluginInfo;
    }

    /**
     * Helper for availability capabilities
     * @return array
     */
    public function getCapabilities(){
        return array(
            'install' => true,
            'update' => true,
            'enable' => true,
            'secureUninstall' => true
        );
    }

    /**
     * Meta info about plugin
     * @return array
     * @throws Exception
     */
    public function getInfo(){
        $this->getPluginDetails();

        if( $this->pluginInfo )
            return array_merge($this->pluginInfo, array(
                'version'   =>  $this->getVersion(),
                'label' =>  $this->getLabel(),
                'source' => $this->getSource(),
                'supplier'    =>  $this->pluginInfo['author'],
                'description'    =>  file_get_contents( __DIR__ . '/description.html' ),
                'support' => $this->pluginInfo['author'],
            ));
        else
            throw new Exception('The plugin has an invalid version file.');
    }

    /**
     * Returns the current version of the plugin.
     *
     * @return string
     * @throws Exception
     */
    public function getVersion()
    {
        $info = json_decode(file_get_contents(__DIR__ . '/plugin.json'), true);

        if ($info) {
            return $info['currentVersion'];
        } else {
            throw new Exception('The plugin has an invalid version file.');
        }
    }

    /**
     * Get (nice) name for plugin manager list
     * @return string
     */
    public function getLabel(){
        return $this->getPluginDetails()['label']['de'];
    }

    /**
     * @return array
     */
    public function getArrayConfig(){
        $config = $this->Config();
        if ($config instanceof \Enlight_Config) {
            $config = $config->toArray();
        } else {
            $config = (array) $config;
        }

        return $config;
    }

    /**
     * Standard plugin install method to register all required components.
     * @return array
     */
    public function install() {
        try {
            if (!$this->assertMinimumVersion('5.0.0')) {
                throw new Exception('This plugin requires Shopware 5.0.0 or a later version.');
            }elseif(
                version_compare($this->Application()->Config()->version, '6.0.0', '>=')
            ){
                throw new Exception(
                    'This plugin supports only Shopware 5, you can download version for Shopware 6 on <a href="https://cookiefirst.com/" target="_blank">cookiefirst</a>'
                );
            }

            $this->createConfiguration();
            $this->subscribeEvent('Enlight_Controller_Action_PostDispatch_Frontend','onPostDispatch');
        } catch(Exception $e) {
            $this->uninstall();
            return array(
                'success' => false,
                'message' => $e->getMessage(),
            );
        }

        return [
            'success' => true,
            'message' => 'Plugin was successfully installed'
        ];
    }

    /**
     * On uninstall remove attributes and re-generate article-details-attribute models
     * @return array
     */
    public function uninstall(){
        return array(
            'success'   =>  true,
            'message'   =>  'Plugin was sucessfully uninstalled',
            'invalidateCache'   =>  $this->getInvalidateCacheArray()
        );
    }

    /**
     * Enable plugin method
     */
    public function enable()
    {
        if ( parent::enable() ){
            $config = $this->getArrayConfig();
            if( empty($config['apiKey']) ){
                return false;
            }

            return array('success' => true, 'invalidateCache' => $this->getInvalidateCacheArray());
        }

        return false;
    }

    /**
     * Creates plugin configuration
     */
    private function createConfiguration()
    {
        $this->Form()->setElement('boolean', 'showWidget', array(
            'value' => true,
            'label' => 'Enable CookieFirst for subshop:',
            'scope' => Shopware\Models\Config\Element::SCOPE_SHOP
        ));
        $this->Form()->setElement('text', 'apiKey', array(
            'label' => 'API KEY',
            'value' =>  '',
            'scope' => Shopware\Models\Config\Element::SCOPE_SHOP,
            'required' => true
        ));
    }

    /**
     * Called when Detail product page open on front-end
     * @param \Enlight_Controller_ActionEventArgs $arguments
     */
    public function onPostDispatch(\Enlight_Controller_ActionEventArgs $arguments){
        /**@var $controller \Shopware_Controllers_Frontend_Checkout */
        $controller = $arguments->getSubject();
        $request  = $controller->Request();
        $response = $controller->Response();
        $view = $controller->View();

        if (
            !$request->isDispatched() || $response->isException() ||
            !$view->hasTemplate() || !$this->Config()->get('showWidget')
        ) {
            return;
        }

        $view->assign('apiKey', $this->Config()->get('apiKey'));
        $hostname = preg_replace('/^www\./', '', Shopware()->Shop()->getHost());
        $view->assign('siteUrl', $hostname);
        $view->addTemplateDir( $this->Path() . 'Views/' . $this->getTemplateDirVersion());
    }

    /**
     * @return string
     */
    public function getTemplateDirVersion(){
        if( Shopware()->Shop()->getTemplate()->getVersion() >= 3 )
            if($this->assertMinimumVersion('5.3.0')){
//          Responsive template, Shopware5.3.0+ - async js
                return 'responsive530/';
            } else {
//          Responsive template, Shopware5 untill Shopware5.3.0
                return 'responsive500/';
            }
        else{
//          Emotion template, Shopware4 and initially versions of Shopware5
            return 'emotion/';
        }
    }
}
